<?php
/**
 * API Endpoint: Verify OTP
 *
 * Completes the redemption process by:
 * 1. Validating OTP format
 * 2. Finding active session
 * 3. Calling Node.js to verify OTP and apply promo code
 * 4. Marking code as used on success
 */

require_once '../config.php';

// Set response headers
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendJSON(['success' => false, 'error' => 'Method not allowed'], 405);
}

// Parse JSON input
$data = json_decode(file_get_contents('php://input'), true);

// Validate input
$otp = isset($data['otp']) && preg_match('/^\d{6}$/', $data['otp']) ? $data['otp'] : null;
$uuid = isset($data['uuid']) && isValidUUID($data['uuid']) ? $data['uuid'] : null;

if (!$otp) {
    sendJSON(['success' => false, 'error' => 'invalid_otp', 'message' => 'OTP must be 6 digits'], 400);
}

if (!$uuid) {
    sendJSON(['success' => false, 'error' => 'invalid_uuid', 'message' => 'Invalid redemption link'], 400);
}

try {
    // Get most recent session for this UUID that's waiting for OTP
    $stmt = $pdo->prepare("
        SELECT r.session_id, r.email, r.created_at, c.code
        FROM redemptions r
        JOIN codes c ON r.uuid = c.uuid
        WHERE r.uuid = ?
        AND r.status = 'otp_sent'
        AND r.created_at > DATE_SUB(NOW(), INTERVAL ? MINUTE)
        ORDER BY r.created_at DESC
        LIMIT 1
    ");
    $stmt->execute([$uuid, SESSION_TIMEOUT_MINUTES]);
    $sessionData = $stmt->fetch();

    if (!$sessionData) {
        sendJSON([
            'success' => false,
            'error' => 'session_expired',
            'message' => 'Session expired or not found. Please start again.'
        ], 404);
    }

    // Check OTP attempts for this session (prevent brute force)
    $stmt = $pdo->prepare("
        SELECT otp, verified
        FROM redemptions
        WHERE session_id = ?
    ");
    $stmt->execute([$sessionData['session_id']]);
    $attemptData = $stmt->fetch();

    // Count failed attempts (if we implement attempt tracking)
    // For now, we'll let Node.js handle the OTP verification

    // Update OTP in database
    $stmt = $pdo->prepare("UPDATE redemptions SET otp = ? WHERE session_id = ?");
    $stmt->execute([$otp, $sessionData['session_id']]);

    // Call Node.js to verify OTP and apply promo code
    $result = callNodeService('/verify-otp', [
        'uuid' => $uuid,
        'otp' => $otp,
        'promo_code' => $sessionData['code'],
        'secret' => NODE_SECRET
    ]);

    if ($result['success']) {
        // Mark code as used
        $stmt = $pdo->prepare("
            UPDATE codes
            SET status = 'used', used_at = NOW(), used_by = ?
            WHERE uuid = ?
        ");
        $stmt->execute([$sessionData['email'], $uuid]);

        // Update redemption status
        $stmt = $pdo->prepare("
            UPDATE redemptions
            SET status = 'activated', verified = TRUE
            WHERE session_id = ?
        ");
        $stmt->execute([$sessionData['session_id']]);

        sendJSON([
            'success' => true,
            'message' => 'Perplexity Pro activated successfully!',
            'callback_url' => $result['callback_url'] ?? 'https://www.perplexity.ai'
        ]);

    } else {
        // Update with error
        $errorMsg = $result['error'] ?? 'Unknown error';
        $stmt = $pdo->prepare("
            UPDATE redemptions
            SET status = 'failed', error_message = ?
            WHERE session_id = ?
        ");
        $stmt->execute([$errorMsg, $sessionData['session_id']]);

        // Provide user-friendly error messages
        $userMessage = 'Verification failed';
        $errorCode = 'verification_failed';

        if (strpos($errorMsg, 'Invalid OTP') !== false || strpos($errorMsg, 'login failed') !== false) {
            $userMessage = 'Invalid OTP code. Please check your email and try again.';
            $errorCode = 'invalid_otp';
        } elseif (strpos($errorMsg, 'already has') !== false || strpos($errorMsg, 'already subscribed') !== false) {
            $userMessage = 'This account already has or had a Pro subscription. Promo codes only work for new users.';
            $errorCode = 'already_subscribed';
        } elseif (strpos($errorMsg, 'promo code invalid') !== false || strpos($errorMsg, 'Activation failed') !== false) {
            $userMessage = 'Promo code could not be applied. It may be invalid or expired.';
            $errorCode = 'promo_invalid';
        } elseif (strpos($errorMsg, 'not found') !== false) {
            $userMessage = 'Session not found. Please start again.';
            $errorCode = 'session_not_found';
        }

        sendJSON([
            'success' => false,
            'error' => $errorCode,
            'message' => $userMessage
        ], 400);
    }

} catch (PDOException $e) {
    error_log("Database error in verify-otp.php: " . $e->getMessage());
    sendJSON([
        'success' => false,
        'error' => 'database_error',
        'message' => 'Database error occurred'
    ], 500);
} catch (Exception $e) {
    error_log("Error in verify-otp.php: " . $e->getMessage());
    sendJSON([
        'success' => false,
        'error' => 'server_error',
        'message' => 'An error occurred'
    ], 500);
}
?>
