<?php
/**
 * API Endpoint: Send OTP
 *
 * Initiates the redemption process by:
 * 1. Validating email and UUID
 * 2. Checking if code is available
 * 3. Calling Node.js automation to send OTP via Perplexity
 */

require_once '../config.php';

// Set response headers
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendJSON(['success' => false, 'error' => 'Method not allowed'], 405);
}

// Parse JSON input
$data = json_decode(file_get_contents('php://input'), true);

// Validate input
$email = validateEmail($data['email'] ?? '');
$uuid = isset($data['uuid']) && isValidUUID($data['uuid']) ? $data['uuid'] : null;

if (!$email) {
    sendJSON(['success' => false, 'error' => 'invalid_email', 'message' => 'Invalid email address'], 400);
}

if (!$uuid) {
    sendJSON(['success' => false, 'error' => 'invalid_uuid', 'message' => 'Invalid redemption link'], 400);
}

try {
    // Check if code exists and is available
    $stmt = $pdo->prepare("SELECT code FROM codes WHERE uuid = ? AND status = 'available'");
    $stmt->execute([$uuid]);
    $codeData = $stmt->fetch();

    if (!$codeData) {
        // Check if already used
        $stmt = $pdo->prepare("
            SELECT used_by, DATE_FORMAT(used_at, '%Y-%m-%d %H:%i') as used_at
            FROM codes
            WHERE uuid = ?
        ");
        $stmt->execute([$uuid]);
        $usedData = $stmt->fetch();

        if ($usedData) {
            sendJSON([
                'success' => false,
                'error' => 'already_used',
                'message' => 'This code has already been used',
                'used_by' => substr($usedData['used_by'], 0, 3) . '***', // Partially hide email
                'used_at' => $usedData['used_at']
            ], 400);
        } else {
            sendJSON([
                'success' => false,
                'error' => 'invalid_code',
                'message' => 'Invalid redemption link'
            ], 404);
        }
    }

    // Check if this email has already attempted this code recently (prevent spam)
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as attempts
        FROM redemptions
        WHERE uuid = ? AND email = ?
        AND created_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
    ");
    $stmt->execute([$uuid, $email]);
    $recentAttempts = $stmt->fetch();

    if ($recentAttempts['attempts'] >= 3) {
        sendJSON([
            'success' => false,
            'error' => 'too_many_attempts',
            'message' => 'Too many attempts. Please wait 5 minutes.'
        ], 429);
    }

    // Generate unique session ID
    $sessionId = bin2hex(random_bytes(32));

    // Store redemption attempt
    $stmt = $pdo->prepare("
        INSERT INTO redemptions (email, uuid, session_id, ip_address, status)
        VALUES (?, ?, ?, ?, 'pending')
    ");
    $stmt->execute([$email, $uuid, $sessionId, $_SERVER['REMOTE_ADDR'] ?? 'unknown']);

    // Call Node.js automation service to start redemption
    $result = callNodeService('/send-otp', [
        'email' => $email,
        'uuid' => $uuid,
        'secret' => NODE_SECRET
    ]);

    if ($result['success']) {
        // Update status to otp_sent
        $stmt = $pdo->prepare("UPDATE redemptions SET status = 'otp_sent' WHERE session_id = ?");
        $stmt->execute([$sessionId]);

        sendJSON([
            'success' => true,
            'sessionId' => $sessionId,
            'message' => 'OTP sent successfully. Check your email.'
        ]);
    } else {
        // Log error
        $errorMsg = $result['error'] ?? 'Unknown error';
        $stmt = $pdo->prepare("UPDATE redemptions SET status = 'failed', error_message = ? WHERE session_id = ?");
        $stmt->execute([$errorMsg, $sessionId]);

        sendJSON([
            'success' => false,
            'error' => 'send_failed',
            'message' => 'Failed to send OTP: ' . $errorMsg
        ], 500);
    }

} catch (PDOException $e) {
    error_log("Database error in send-otp.php: " . $e->getMessage());
    sendJSON([
        'success' => false,
        'error' => 'database_error',
        'message' => 'Database error occurred'
    ], 500);
} catch (Exception $e) {
    error_log("Error in send-otp.php: " . $e->getMessage());
    sendJSON([
        'success' => false,
        'error' => 'server_error',
        'message' => 'An error occurred'
    ], 500);
}
?>